/**
 * Copyright 2016 VMware, Inc. All rights reserved. -- VMware Confidential
 */
package com.vmware.automatic.plugin.registration.commands;

import com.vmware.automatic.plugin.registration.commands.utils.PrintHelper;
import com.vmware.vim25.Description;
import com.vmware.vim25.Extension;
import com.vmware.vim25.ExtensionClientInfo;
import com.vmware.vim25.ExtensionResourceInfo;
import com.vmware.vim25.ExtensionServerInfo;
import com.vmware.vim25.KeyValue;
import com.vmware.vim25.NotFoundFaultMsg;
import com.vmware.vim25.RuntimeFaultFaultMsg;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;

import javax.xml.datatype.DatatypeConfigurationException;
import javax.xml.datatype.DatatypeFactory;
import javax.xml.datatype.XMLGregorianCalendar;
import java.util.GregorianCalendar;
import java.util.TimeZone;

public class UpdatePluginCmd extends PluginCmd {
   /**
    * Extension server protocol must match to plugin url protocol
    */
   private static final String HTTPS_PROTOCOL = "HTTPS";

   private String _pluginUrl;
   private String _name;
   private String _summary;
   private String _version;
   private String _company;
   private boolean _showInSolutionManager;
   private String _serverThumbprint;

   public UpdatePluginCmd() {
      _action = "-action updatePlugin";
      _errorMessage = "Update plugin to the VC";
   }

   public Options getPluginOpts() {
      Options options = super.getPluginOpts();
      options.addOption(Option.builder("pu").longOpt("pluginUrl").hasArg().argName("plugin url")
            .desc("Url from where the plugin will be downloaded").required().build());
      options.addOption(Option.builder("v").longOpt("version").hasArg().argName("version").desc("Plugin version")
            .required().build());
      options.addOption(Option.builder("n").longOpt("name").hasArg().argName("name").desc("Plugin name")
            .required(false).build());
      options.addOption(Option.builder("s").longOpt("summary").hasArg().argName("summary").desc("Plugin summary")
            .required(false).build());
      options.addOption(Option.builder("c").longOpt("company").hasArg().argName("company").desc("Company")
            .required(false).build());
      options.addOption(Option.builder("st").longOpt("serverThumbprint").hasArg().argName("server thumbprint")
            .desc("Thumbprint of the server from which the plugin will be downloaded.").required(false).build());
      options.addOption(Option.builder("show").longOpt("showInSolutionManager").desc("Summary").required(false)
            .build());
      return options;
   }

   protected void populateFields(String[] commandLineArgs) {
      super.populateFields(commandLineArgs);
      if (_commandLine != null) {
         _pluginUrl = _commandLine.getOptionValue("pu");
         String nameCmd = _commandLine.getOptionValue("n");
         if (nameCmd != null) {
            _name = nameCmd;
         }
         String summary = _commandLine.getOptionValue("s");
         if (summary != null) {
            _summary = summary;
         }
         String version = _commandLine.getOptionValue("v");
         if (version != null) {
            _version = version;
         }
         String company = _commandLine.getOptionValue("c");
         if (company != null) {
            _company = company;
         }
         _showInSolutionManager = _commandLine.hasOption("show");
         String serverThumbprint = _commandLine.getOptionValue("st");
         if (serverThumbprint != null) {
            _serverThumbprint = serverThumbprint;
         }
      }
   }


   /**
    * Registers a single extension based on the set member fields.
    */
   protected void doExecute() {
      try {
         Extension extension = _vimPort.findExtension(_extensionManager, _key);
         if (extension == null) {
            System.out.println("Plugin: " + _key + " is not registered with vCenter <" + _url + ">.");
            return;
         }

         Description description = extension.getDescription();
         if (description == null) {
            description = new Description();
         }
         description.setLabel((_name != null) ? _name : "");
         description.setSummary((_summary != null) ? _summary : "");

         extension.setKey(_key);
         extension.setVersion((_version != null) ? _version : "");
         extension.setCompany(_company);
         extension.setDescription(description);


         ExtensionClientInfo extClientInfo = (extension.getClient().size() == 0) ? new ExtensionClientInfo()
               : extension.getClient().get(0);

         extClientInfo.setVersion((_version != null) ? _version : "");
         extClientInfo.setCompany((_company != null) ? _company : "");
         extClientInfo.setDescription(description);
         extClientInfo.setUrl(_pluginUrl);
         if (extension.getClient().size() == 0) {
            extension.getClient().add(extClientInfo);
         } else {
            extension.getClient().set(0, extClientInfo);
         }

         ExtensionResourceInfo extResourceInfo_en = (extension.getResourceList().size() == 0) ? new ExtensionResourceInfo()
               : extension.getResourceList().get(0);
         extResourceInfo_en.setLocale("en_US");
         extResourceInfo_en.setModule("name");
         KeyValue kv1 = (extResourceInfo_en.getData().size() == 0) ? new KeyValue() : extResourceInfo_en.getData().get(0);
         kv1.setKey("name");
         kv1.setValue((_name != null) ? _name : "");
         if (extension.getResourceList().size() == 0) {
            extResourceInfo_en.getData().add(kv1);
            extension.getResourceList().add(extResourceInfo_en);
         } else {
            if (extResourceInfo_en.getData().size() == 0) {
               extResourceInfo_en.getData().add(kv1);
            } else {
               extResourceInfo_en.getData().set(0, kv1);
            }
            extension.getResourceList().set(0, extResourceInfo_en);
         }

         if (_pluginUrl.startsWith(HTTPS_PROTOCOL.toLowerCase())) {
            // HTTPS requests require server info
            if (_serverThumbprint == null) {
               PrintHelper.printHelp(getPluginOpts(), "Missing required option: -serverThumbprint required argument when -pluginUrl is https");
               return;
            }
            ExtensionServerInfo extServerInfo = (extension.getServer().size() == 0) ? new ExtensionServerInfo()
                  : extension.getServer().get(0);
            extServerInfo.setCompany((_company != null) ? _company : "");
            extServerInfo.setDescription(description);
            extServerInfo.setType(HTTPS_PROTOCOL);
            extServerInfo.setServerThumbprint(_serverThumbprint);
            extServerInfo.setUrl(_pluginUrl);
            if (extension.getServer().size() == 0) {
               extension.getServer().add(extServerInfo);
            } else {
               extension.getServer().set(0, extServerInfo);
            }
         } else {
            System.out.println("INFO: Not using https for your plugin URL is OK for testing but not recommended for production." +
                  "\nUsers will have to include the flag allowHttp=true in their vSphere Client webclient.properties otherwise the http URL will be ignored");
         }
         extension.setShownInSolutionManager(_showInSolutionManager);
         GregorianCalendar cal = new GregorianCalendar(TimeZone.getTimeZone("GMT"));
         DatatypeFactory dtFactory = DatatypeFactory.newInstance();
         XMLGregorianCalendar xmlCalendar = dtFactory.newXMLGregorianCalendar(cal);
         extension.setLastHeartbeatTime(xmlCalendar);
         _vimPort.updateExtension(_extensionManager, extension);
         System.out.println("Plugin: " + _key + " has been successfully updated in vCenter <" + _url + ">.");
      } catch (RuntimeFaultFaultMsg | NotFoundFaultMsg | DatatypeConfigurationException e) {
         System.out.println("Exception while updating plugin" + _key);
         e.printStackTrace();
      }
   }
}
